<?php

namespace Oro\Bundle\SidebarBundle\Controller\Api\Rest;

use FOS\RestBundle\Controller\Annotations\NamePrefix;
use FOS\RestBundle\Controller\Annotations\RouteResource;
use FOS\RestBundle\Controller\FOSRestController;
use FOS\RestBundle\Util\Codes;
use Nelmio\ApiDocBundle\Annotation\ApiDoc;

use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\User\UserInterface;

use Oro\Bundle\SidebarBundle\Entity\Repository\WidgetRepository;
use Oro\Bundle\SidebarBundle\Entity\Widget;

/**
 * @RouteResource("sidebarwidgets")
 * @NamePrefix("oro_api_")
 */
class WidgetController extends FOSRestController
{
    /**
     * REST GET list
     *
     * @ApiDoc(
     *      description="Get all sidebar widget items",
     *      resource=true
     * )
     * @param string $placement
     * @return Response
     */
    public function cgetAction($placement)
    {
        /** @var WidgetRepository $widgetRepository */
        $widgetRepository = $this->getDoctrine()->getRepository('OroSidebarBundle:Widget');
        $items = $widgetRepository->getWidgets(
            $this->getUser(),
            $placement,
            $this->container->get('security.context')->getToken()->getOrganizationContext()
        );

        if (!$items) {
            $items = [];
        }

        return $this->handleView(
            $this->view($items, Codes::HTTP_OK)
        );
    }

    /**
     * REST POST
     *
     * @ApiDoc(
     *  description="Add Sidebar Widget",
     *  resource=true
     * )
     * @return Response
     */
    public function postAction()
    {
        $entity = new Widget();
        $entity->setWidgetName($this->getRequest()->get('widgetName'));
        $entity->setPosition($this->getRequest()->get('position'));
        $entity->setSettings($this->getRequest()->get('settings'));
        $entity->setPlacement($this->getRequest()->get('placement'));
        $entity->setState($this->getRequest()->get('state'));
        $entity->setUser($this->getUser());
        $entity->setOrganization(
            $this->container->get('security.context')->getToken()->getOrganizationContext()
        );

        $manager = $this->getManager();
        $manager->persist($entity);
        $manager->flush();

        return $this->handleView(
            $this->view(['id' => $entity->getId()], Codes::HTTP_CREATED)
        );
    }

    /**
     * REST PUT
     *
     * @param int $widgetId Widget instance id
     *
     * @ApiDoc(
     *  description="Update Sidebar Widget",
     *  resource=true
     * )
     * @return Response
     */
    public function putAction($widgetId)
    {
        /** @var \Oro\Bundle\SidebarBundle\Entity\Widget $entity */
        $entity = $this->getManager()->find('OroSidebarBundle:Widget', (int)$widgetId);
        if (!$entity) {
            return $this->handleView($this->view([], Codes::HTTP_NOT_FOUND));
        }
        if (!$this->validatePermissions($entity->getUser())) {
            return $this->handleView($this->view(null, Codes::HTTP_FORBIDDEN));
        }

        $entity->setState($this->getRequest()->get('state', $entity->getState()));
        $entity->setPosition($this->getRequest()->get('position', $entity->getPosition()));
        $entity->setSettings($this->getRequest()->get('settings', $entity->getSettings()));
        $entity->setPlacement($this->getRequest()->get('placement', $entity->getPlacement()));

        $em = $this->getManager();
        $em->persist($entity);
        $em->flush();

        return $this->handleView($this->view([], Codes::HTTP_OK));
    }

    /**
     * REST DELETE
     *
     * @param int $widgetId
     *
     * @ApiDoc(
     *  description="Remove Sidebar Widget instance",
     *  resource=true
     * )
     * @return Response
     */
    public function deleteAction($widgetId)
    {
        /** @var \Oro\Bundle\SidebarBundle\Entity\Widget $entity */
        $entity = $this->getManager()->find('OroSidebarBundle:Widget', (int)$widgetId);
        if (!$entity) {
            return $this->handleView($this->view([], Codes::HTTP_NOT_FOUND));
        }
        if (!$this->validatePermissions($entity->getUser())) {
            return $this->handleView($this->view(null, Codes::HTTP_FORBIDDEN));
        }

        $em = $this->getManager();
        $em->remove($entity);
        $em->flush();

        return $this->handleView($this->view([], Codes::HTTP_NO_CONTENT));
    }

    /**
     * Validate permissions
     *
     * @param UserInterface $user
     * @return bool
     */
    protected function validatePermissions(UserInterface $user)
    {
        return $user->getUsername() === $this->getUser()->getUsername();
    }

    /**
     * Get entity Manager
     *
     * @return \Doctrine\Common\Persistence\ObjectManager
     */
    protected function getManager()
    {
        return $this->getDoctrine()->getManagerForClass('OroSidebarBundle:Widget');
    }
}
