<?php

namespace Oro\Bundle\EmailBundle\Mailer;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\DependencyInjection\IntrospectableContainerInterface;
use Oro\Bundle\EmailBundle\Exception\NotSupportedException;
use \Swift_SmtpTransport;
use Oro\Bundle\SecurityBundle\Encoder\Mcrypt;

/**
 * The goal of this class is to send an email directly, not using a mail spool
 * even when it is configured for a base mailer
 */
class DirectMailer extends \Swift_Mailer
{
    /**
     * @var \Swift_Mailer
     */
    protected $baseMailer;

    /**
     * @var ContainerInterface
     */
    protected $container;

    protected $transport;

    protected $encryptor;

    /**
     * Constructor
     *
     * @param \Swift_Mailer      $baseMailer
     * @param ContainerInterface $container
     */
    public function __construct(\Swift_Mailer $baseMailer, ContainerInterface $container, Mcrypt $encryptor)
    {

        $this->container  = $container;
        $this->encryptor = $encryptor;

        $emailConfig = $this->getEmailConfig($this->getUser());

        if($emailConfig){
            $this->transport = \Swift_SmtpTransport::newInstance('smtp.gmail.com', 465)
                ->setUsername($emailConfig->getUser())
                ->setPassword($this->encryptor->decryptData($emailConfig->getPassword()))
                ->setEncryption('ssl');
            $this->baseMailer = \Swift_Mailer::newInstance($this->transport);
        } else {
            $this->baseMailer = $baseMailer;
            $this->transport = $this->baseMailer->getTransport();
        }



        //$this->baseMailer = $baseMailer;//\Swift_Mailer::newInstance($this->transport);

        //s$this->transport = $this->baseMailer->getTransport();
        if ($this->transport instanceof \Swift_Transport_SpoolTransport) {
            $this->transport = $this->findRealTransport();
            if (!$this->transport) {
                $this->transport = \Swift_NullTransport::newInstance();
            }
        }
        parent::__construct($this->transport);
    }

    /**
     * Register a plugin using a known unique key (e.g. myPlugin).
     *
     * @param \Swift_Events_EventListener $plugin
     * @throws \Oro\Bundle\EmailBundle\Exception\NotSupportedException
     */
    public function registerPlugin(\Swift_Events_EventListener $plugin)
    {
        throw new NotSupportedException('The registerPlugin() is not supported for this mailer.');
    }

    /**
     * Sends the given message.
     *
     * The return value is the number of recipients who were accepted for
     * delivery.
     *
     * @param \Swift_Mime_Message $message
     * @param array               $failedRecipients An array of failures by-reference
     *
     * @return int The number of recipients who were accepted for delivery
     * @throws \Exception
     */
    public function send(\Swift_Mime_Message $message, &$failedRecipients = null)
    {
        $result = 0;
        // start a transport if needed
        $needToStopRealTransport = false;
        if (!$this->getTransport()->isStarted()) {
            $this->getTransport()->start();
            $needToStopRealTransport = true;
        }
        // send a mail
        $sendException = null;
        try {
            $result = parent::send($message, $failedRecipients);
        } catch (\Exception $unexpectedEx) {
            $sendException = $unexpectedEx;
        }
        // stop a transport if it was started before
        if ($needToStopRealTransport) {
            try {
                $this->getTransport()->stop();
            } catch (\Exception $ex) {
                // ignore errors here
            }
        }
        // rethrow send failure
        if ($sendException) {
            throw $sendException;
        }

        return $result;
    }

    /**
     * Get a user from the Security Context
     *
     * @return mixed
     *
     * @throws \LogicException If SecurityBundle is not available
     *
     * @see Symfony\Component\Security\Core\Authentication\Token\TokenInterface::getUser()
     */
    protected function getUser()
    {
        if (!$this->container->has('security.context')) {
            throw new \LogicException('The SecurityBundle is not registered in your application.');
        }

        if (null === $token = $this->container->get('security.context')->getToken()) {
            return;
        }

        if (!is_object($user = $token->getUser())) {
            return;
        }

        return $user;
    }

    /**
     *
     * @param \Oro\Bundle\UserBundle\Entity\User $user
     * @return \Oro\Bundle\ImapBundle\Entity\ImapEmailOrigin
     */
    protected function getEmailConfig($user)
    {
        if($user->getImapConfiguration())
        {
            return $user->getImapConfiguration();

        } else {

            return null;
        }

        //return $user->getImapConfiguration();

    }

    /**
     * Returns a real transport used to send mails by a mailer specified in the constructor of this class
     *
     * @return \Swift_Transport|null
     */
    protected function findRealTransport()
    {
        $realTransport = null;
        $mailers       = array_keys($this->container->getParameter('swiftmailer.mailers'));
        foreach ($mailers as $name) {
            if ($this->container instanceof IntrospectableContainerInterface
                && !$this->container->initialized(sprintf('swiftmailer.mailer.%s', $name))
            ) {
                continue;
            }
            $mailer = $this->container->get(sprintf('swiftmailer.mailer.%s', $name));
            if ($mailer === $this->baseMailer) {
                $realTransport = $this->container->get(sprintf('swiftmailer.mailer.%s.transport.real', $name));
                break;
            }
        }

        return $realTransport;
    }
}
