<?php

namespace SoundView\Bundle\CommissionBundle\Controller;

use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;

use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;

use Oro\Bundle\SecurityBundle\Annotation\Acl;
use Oro\Bundle\SecurityBundle\Annotation\AclAncestor;

use Oro\Bundle\UserBundle\Entity\User;
use OroCRM\Bundle\AccountBundle\Entity\Account;
use OroCRM\Bundle\ContactBundle\Entity\Contact;
use SoundView\Bundle\SaleBundle\Entity\Sale;
use SoundView\Bundle\CommissionBundle\Entity\CommissionStatus;
use SoundView\Bundle\CommissionBundle\Entity\Commission;




class CommissionController extends Controller
{
    /**
     * @Route(
     *      ".{_format}",
     *      name="svwd_sale_commission_index",
     *      requirements={"_format"="html|json"},
     *      defaults={"_format" = "html"}
     * )
     * @Acl(
     *      id="svwd_sale_commission_view",
     *      type="entity",
     *      class="SvwdCommissionBundle:Commission",
     *      permission="VIEW"
     * )
     * @Template
     */
    public function indexAction()
    {
        return [
            'entity_class' => $this->container->getParameter('svwd_sale.commission.class')
        ];
    }
    
     /**
     * @Route("/info/{id}", name="svwd_sale_commission_info", requirements={"id"="\d+"})
     *
     * @Template
     * @AclAncestor("svwd_sale_commission_view")
     */
    public function infoAction(Commission $commission)
    {

        return array(
            'entity'  => $commission
        );
    }
    
        /**
     * @Route("/widget/sale-commission/{id}", name="svwd_sale_widget_sale_commission", requirements={"id"="\d+"})
     * @AclAncestor("svwd_sale_commission_view")
     * @Template
     */
    public function saleCommissionAction(Sale $sale)
    {
        $calc = $this->getCalculator();
        $commission = $calc->setSale($sale);
        
        
        
        return array('entity' => $sale, 'commission' => $commission);
    }
    



/**
     * @Route("/create/{saleId}", 
     * name="svwd_sale_commission_create",
     *      requirements={"saleId"="\d+"}
     * )
     * @Acl(
     *      id="svwd_sale_commission_create",
     *      type="entity",
     *      class="SvwdCommissionBundle:Commission",
     *      permission="CREATE"
     * )
     * @Template("SvwdCommissionBundle:Commission:update.html.twig")
     * @ParamConverter("sale", options={"id" = "saleId"})
     */
    public function createAction(Sale $sale)
    {
        $commission = $this->initEntity($sale);

        return $this->update($commission);
    }
    
    /**
     * @param Sale $sale
     * @return Commission
     * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
     */
    protected function initEntity(Sale $sale)
    {
        $entity = new Commission();

        $entity->setSale($sale);
        $entity->setOwner($sale->getOwner());
        $entity->setRelatedAccount($sale->getRelatedAccount());
        $entity->setRefNumber($this->generateRefNumber($sale));

        return $entity;
    }


    /**
     * @Route("/create/{saleId}", 
     * name="svwd_sale_commission_create_adjustment",
     *      requirements={"saleId"="\d+"}
     * )
     * @Acl(
     *      id="svwd_sale_commission_create",
     *      type="entity",
     *      class="SvwdCommissionBundle:Commission",
     *      permission="CREATE"
     * )
     * @Template("SvwdCommissionBundle:Commission:update.html.twig")
     * @ParamConverter("sale", options={"id" = "saleId"})
     */
    public function createAdjustmentAction(Sale $sale)
    {
        $typeRepo = $this->getRepository('SvwdCommissionBundle:CommissionStatus');
        $type = $typeRepo->findOneByName('adjustment');

        $commission = new Commission();

        $commission->setType($type);
        $entity->setSale($sale);
        $entity->setOwner($sale->getOwner());
        $entity->setRelatedAccount($sale->getRelatedAccount());

        return $this->update($commission);
    }
    
    

    /**
     * @return User
     */
    public function getCurrentUser()
    {
        $token = $this->container->get('security.context')->getToken();

        return $token ? $token->getUser() : null;
    }
    
    protected function generateRefNumber(Sale $sale)
    {
        $i = substr($sale->getOwner()->getFirstName(), 0, 1);
        $i .= substr($sale->getOwner()->getLastName(), 0, 1);
        
        $so = $sale->getSoNumber();
        
        $count = $sale->getCommissions()->count();
        $num = strtoupper(chr(($count % 26) + 97));
        
        $ref = $i;
        $ref .= $so;
        $ref .= $num;
        
        return (string) trim($ref);
        
        
    }


    /**
     * @Route("/view/{id}", name="svwd_sale_commission_view", requirements={"id"="\d+"})
     *
     * @Template
     * @Acl(
     *      id="svwd_sale_commission_view",
     *      type="entity",
     *      permission="VIEW",
     *      class="SvwdCommissionBundle:Commission"
     * )
     */
    public function viewAction(Commission $commission)
    {


        return [
            'entity' => $commission
        ];
    }

    /**
     * @Route("/update/{id}", name="svwd_sale_commission_update", requirements={"id"="\d+"})
     * @Template
     * @Acl(
     *      id="svwd_sale_commission_update",
     *      type="entity",
     *      class="SvwdCommissionBundle:Commission",
     *      permission="EDIT"
     * )
     */
    public function updateAction(Commission $commission)
    {
        return $this->update($commission);
    }
    
    /**
     * @Route("/pay/{id}", name="svwd_sale_commission_pay", requirements={"id"="\d+"})
     * @Template("SvwdCommissionBundle:Commission/widget:pay.html.twig")
     * @AclAncestor("svwd_sale_commission_update")
     */
    public function payAction(Commission $commission)
    {
        $statusRepo = $this->getRepository('SvwdCommissionBundle:CommissionStatus');
        //$commission->createRefNumber();
        $commission->setDue(false);
        $commission->setPaid(true);
        $commission->setPaidDate(new \DateTime('now', new \DateTimeZone('UTC')));
        $commission->setStatus($statusRepo->findOneByName('paid'));
        
        return $this->pay($commission);
    }
    
    
    /**
     * @Route("/schedule/{id}", name="svwd_sale_commission_schedule", requirements={"id"="\d+"})
     * @Template("SvwdCommissionBundle:Commission/widget:schedule.html.twig")
     * @AclAncestor("svwd_sale_commission_update")
     */
    public function scheduleAction(Commission $commission)
    {
        $statusRepo = $this->getRepository('SvwdCommissionBundle:CommissionStatus');
        $status = $statusRepo->findOneByName('scheduled');
        
        $commission->createRefNumber();
        $commission->setDue(true);
        $commission->setStatus($statusRepo->findOneByName('scheduled'));
        
        return $this->schedule($commission);
    }
    
        /**
     * @param Commission $commission
     * @return array
     */
    protected function pay(Commission $entity)
    {
        
        $savedId = false;
        $request = $this->getRequest();
        $form = $this->createForm($this->get('svwd_sale.commission_pay.form.type'), $entity);

        if ($request->isMethod('POST')) {
            $form->submit($request);
            if ($form->isValid()) {
                $this->getDoctrine()->getManager()->persist($entity);
                $this->getDoctrine()->getManager()->flush();

                $savedId =  true;

                if (!$this->getRequest()->request->has('_widgetContainer')) {
                    $this->get('session')->getFlashBag()->add(
                        'success',
                        'Commission Paid'
                    );

                    return $this->get('oro_ui.router')->redirectAfterSave(
                        array(
                            'route' => 'svwd_sale_commission_pay',
                            'parameters' => array('id' => $entity->getId()),
                        ),
                        array(
                            'route' => 'svwd_sale_commission_view',
                            'parameters' => array('id' => $entity->getId()),
                        )
                    );
                }
            }
        }

        return array(
            'savedId' => $savedId,
            'entity' => $entity,
            'form' => $form->createView()
        );
    }
    
            /**
     * @param Commission $commission
     * @return array
     */
    protected function schedule(Commission $entity)
    {
        
        $savedId = false;
        $request = $this->getRequest();
        $form = $this->createForm($this->get('svwd_sale.commission.schedule.form.type'), $entity);

        if ($request->isMethod('POST')) {
            $form->submit($request);
            if ($form->isValid()) {
                $this->getDoctrine()->getManager()->persist($entity);
                $this->getDoctrine()->getManager()->flush();

                $savedId =  true;

                if (!$this->getRequest()->request->has('_widgetContainer')) {
                    $this->get('session')->getFlashBag()->add(
                        'success',
                        'Commission Paid'
                    );

                    return $this->get('oro_ui.router')->redirectAfterSave(
                        array(
                            'route' => 'svwd_sale_commission_schedule',
                            'parameters' => array('id' => $entity->getId()),
                        ),
                        array(
                            'route' => 'svwd_sale_commission_view',
                            'parameters' => array('id' => $entity->getId()),
                        )
                    );
                }
            }
        }

        return array(
            'savedId' => $savedId,
            'entity' => $entity,
            'form' => $form->createView()
        );
    }

    /**
     * @param Commission $commission
     * @return array
     */
    protected function update(Commission $entity = null)
    {
        
        if (!$entity) {
            $entity = $this->getManager()->createEntity();
        }

        return $this->get('oro_form.model.update_handler')->handleUpdate(
            $entity,
            $form = $this->createForm($this->getFormType(), $entity),
            function (Commission $entity) {
                return array(
                    'route' => 'svwd_sale_commission_update',
                    'parameters' => array('id' => $entity->getId())
                );
            },
            function (Commission $entity) {
                return array(
                    'route' => 'svwd_sale_commission_view',
                    'parameters' => array('id' => $entity->getId())
                );
            },
            'Commission Saved',
            $this->get('svwd_sale.commission.form.handler.api')
        );
    }

    /**
     * @return SaleType
     */
    protected function getFormType()
    {
        return $this->get('svwd_sale.commission.form.type.api');
    }
    
    public function getCalculator()
    {
        return $this->get('svwd_sale.calculator');
        
    
    }
    
        /**
     * @param string $entityName
     * @return \Doctrine\Common\Persistence\ObjectRepository
     */
    protected function getRepository($entityName)
    {
        return $this->getDoctrine()->getRepository($entityName);
    }
    
}
